﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/glue/GlueRequest.h>
#include <aws/glue/Glue_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Glue {
namespace Model {

/**
 */
class UpdateColumnStatisticsTaskSettingsRequest : public GlueRequest {
 public:
  AWS_GLUE_API UpdateColumnStatisticsTaskSettingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateColumnStatisticsTaskSettings"; }

  AWS_GLUE_API Aws::String SerializePayload() const override;

  AWS_GLUE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the database where the table resides.</p>
   */
  inline const Aws::String& GetDatabaseName() const { return m_databaseName; }
  inline bool DatabaseNameHasBeenSet() const { return m_databaseNameHasBeenSet; }
  template <typename DatabaseNameT = Aws::String>
  void SetDatabaseName(DatabaseNameT&& value) {
    m_databaseNameHasBeenSet = true;
    m_databaseName = std::forward<DatabaseNameT>(value);
  }
  template <typename DatabaseNameT = Aws::String>
  UpdateColumnStatisticsTaskSettingsRequest& WithDatabaseName(DatabaseNameT&& value) {
    SetDatabaseName(std::forward<DatabaseNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the table for which to generate column statistics.</p>
   */
  inline const Aws::String& GetTableName() const { return m_tableName; }
  inline bool TableNameHasBeenSet() const { return m_tableNameHasBeenSet; }
  template <typename TableNameT = Aws::String>
  void SetTableName(TableNameT&& value) {
    m_tableNameHasBeenSet = true;
    m_tableName = std::forward<TableNameT>(value);
  }
  template <typename TableNameT = Aws::String>
  UpdateColumnStatisticsTaskSettingsRequest& WithTableName(TableNameT&& value) {
    SetTableName(std::forward<TableNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The role used for running the column statistics.</p>
   */
  inline const Aws::String& GetRole() const { return m_role; }
  inline bool RoleHasBeenSet() const { return m_roleHasBeenSet; }
  template <typename RoleT = Aws::String>
  void SetRole(RoleT&& value) {
    m_roleHasBeenSet = true;
    m_role = std::forward<RoleT>(value);
  }
  template <typename RoleT = Aws::String>
  UpdateColumnStatisticsTaskSettingsRequest& WithRole(RoleT&& value) {
    SetRole(std::forward<RoleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A schedule for running the column statistics, specified in CRON syntax.</p>
   */
  inline const Aws::String& GetSchedule() const { return m_schedule; }
  inline bool ScheduleHasBeenSet() const { return m_scheduleHasBeenSet; }
  template <typename ScheduleT = Aws::String>
  void SetSchedule(ScheduleT&& value) {
    m_scheduleHasBeenSet = true;
    m_schedule = std::forward<ScheduleT>(value);
  }
  template <typename ScheduleT = Aws::String>
  UpdateColumnStatisticsTaskSettingsRequest& WithSchedule(ScheduleT&& value) {
    SetSchedule(std::forward<ScheduleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of column names for which to run statistics.</p>
   */
  inline const Aws::Vector<Aws::String>& GetColumnNameList() const { return m_columnNameList; }
  inline bool ColumnNameListHasBeenSet() const { return m_columnNameListHasBeenSet; }
  template <typename ColumnNameListT = Aws::Vector<Aws::String>>
  void SetColumnNameList(ColumnNameListT&& value) {
    m_columnNameListHasBeenSet = true;
    m_columnNameList = std::forward<ColumnNameListT>(value);
  }
  template <typename ColumnNameListT = Aws::Vector<Aws::String>>
  UpdateColumnStatisticsTaskSettingsRequest& WithColumnNameList(ColumnNameListT&& value) {
    SetColumnNameList(std::forward<ColumnNameListT>(value));
    return *this;
  }
  template <typename ColumnNameListT = Aws::String>
  UpdateColumnStatisticsTaskSettingsRequest& AddColumnNameList(ColumnNameListT&& value) {
    m_columnNameListHasBeenSet = true;
    m_columnNameList.emplace_back(std::forward<ColumnNameListT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The percentage of data to sample.</p>
   */
  inline double GetSampleSize() const { return m_sampleSize; }
  inline bool SampleSizeHasBeenSet() const { return m_sampleSizeHasBeenSet; }
  inline void SetSampleSize(double value) {
    m_sampleSizeHasBeenSet = true;
    m_sampleSize = value;
  }
  inline UpdateColumnStatisticsTaskSettingsRequest& WithSampleSize(double value) {
    SetSampleSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Data Catalog in which the database resides.</p>
   */
  inline const Aws::String& GetCatalogID() const { return m_catalogID; }
  inline bool CatalogIDHasBeenSet() const { return m_catalogIDHasBeenSet; }
  template <typename CatalogIDT = Aws::String>
  void SetCatalogID(CatalogIDT&& value) {
    m_catalogIDHasBeenSet = true;
    m_catalogID = std::forward<CatalogIDT>(value);
  }
  template <typename CatalogIDT = Aws::String>
  UpdateColumnStatisticsTaskSettingsRequest& WithCatalogID(CatalogIDT&& value) {
    SetCatalogID(std::forward<CatalogIDT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Name of the security configuration that is used to encrypt CloudWatch
   * logs.</p>
   */
  inline const Aws::String& GetSecurityConfiguration() const { return m_securityConfiguration; }
  inline bool SecurityConfigurationHasBeenSet() const { return m_securityConfigurationHasBeenSet; }
  template <typename SecurityConfigurationT = Aws::String>
  void SetSecurityConfiguration(SecurityConfigurationT&& value) {
    m_securityConfigurationHasBeenSet = true;
    m_securityConfiguration = std::forward<SecurityConfigurationT>(value);
  }
  template <typename SecurityConfigurationT = Aws::String>
  UpdateColumnStatisticsTaskSettingsRequest& WithSecurityConfiguration(SecurityConfigurationT&& value) {
    SetSecurityConfiguration(std::forward<SecurityConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_databaseName;

  Aws::String m_tableName;

  Aws::String m_role;

  Aws::String m_schedule;

  Aws::Vector<Aws::String> m_columnNameList;

  double m_sampleSize{0.0};

  Aws::String m_catalogID;

  Aws::String m_securityConfiguration;
  bool m_databaseNameHasBeenSet = false;
  bool m_tableNameHasBeenSet = false;
  bool m_roleHasBeenSet = false;
  bool m_scheduleHasBeenSet = false;
  bool m_columnNameListHasBeenSet = false;
  bool m_sampleSizeHasBeenSet = false;
  bool m_catalogIDHasBeenSet = false;
  bool m_securityConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
